﻿using log4net;
using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Http.Features;
using Microsoft.Extensions.Configuration;
using System;
using System.Linq;
using System.Threading.Tasks;

namespace VA.PPMS.IWS.Api.Middleware
{
    public class ClientCertificateValidator
    {
        private readonly RequestDelegate _next;
        private readonly ILog _logger;
        private readonly IConfiguration _configuration;

        public ClientCertificateValidator(RequestDelegate next, ILog logger, IConfiguration configuration)
        {
            _next = next;
            _logger = logger;
            _configuration = configuration;
        }

        public Task Invoke(HttpContext httpContext)
        {
            var tlsConnectionFeature = httpContext.Features.Get<ITlsConnectionFeature>();

            if (tlsConnectionFeature == null)
            {
                _logger.Error("********* Error: TLS Connection Feature is null *********");
                httpContext.Response.StatusCode = StatusCodes.Status401Unauthorized;
                return Task.FromResult(0);
            }

            var certificate = tlsConnectionFeature.GetClientCertificateAsync(httpContext.RequestAborted).GetAwaiter().GetResult();

            if (certificate == null)
            {
                _logger.Error("********* Error: Certificate is null *********");
                httpContext.Response.StatusCode = StatusCodes.Status401Unauthorized;
                return Task.FromResult(0);
            }

            if (DateTime.UtcNow < certificate.NotBefore || DateTime.UtcNow > certificate.NotAfter)
            {
                _logger.Error("********* Error: Certificate Expired *********");
                httpContext.Response.StatusCode = StatusCodes.Status401Unauthorized;
                return Task.FromResult(0);
            }

            var clientCertNamesSection = _configuration.GetSection("ClientCerts");
            var clientCerts = clientCertNamesSection.GetChildren();

            if (clientCerts.Any(clientCert => certificate.Subject.ToLower().Contains(clientCert.Value)))
            {
                return _next(httpContext);
            }

            _logger.Error("********* Error: Wrong Cert *********");
            httpContext.Response.StatusCode = StatusCodes.Status401Unauthorized;
            return Task.FromResult(0);
        }
    }

    public static class CertCheckerExtensions
    {
        public static IApplicationBuilder UseClientCertificateValidator(this IApplicationBuilder builder)
        {
            return builder.UseMiddleware<ClientCertificateValidator>();
        }
    }
}